#!/bin/bash

# Pomo CLI Installation Script
# Installs pomo-cli-integrated with local SQLite and API sync

set -e

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Print colored output
print_status() {
    echo -e "${BLUE}[INFO]${NC} $1"
}

print_success() {
    echo -e "${GREEN}[SUCCESS]${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}[WARNING]${NC} $1"
}

print_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

# Check if Go is installed
check_go() {
    if ! command -v go &> /dev/null; then
        print_error "Go is not installed. Please install Go 1.21 or later."
        echo "Visit: https://golang.org/dl/"
        exit 1
    fi
    
    GO_VERSION=$(go version | awk '{print $3}' | sed 's/go//')
    print_status "Found Go version: $GO_VERSION"
}

# Check if SQLite is available
check_sqlite() {
    if ! command -v sqlite3 &> /dev/null; then
        print_warning "SQLite3 not found. Installing..."
        if command -v apt-get &> /dev/null; then
            sudo apt-get update && sudo apt-get install -y sqlite3
        elif command -v yum &> /dev/null; then
            sudo yum install -y sqlite
        elif command -v brew &> /dev/null; then
            brew install sqlite3
        else
            print_warning "Could not install SQLite automatically. Please install it manually."
        fi
    fi
}

# Install pomo-cli
install_pomo_cli() {
    print_status "Installing pomo-cli-integrated..."
    
    # Create temporary directory
    TMP_DIR=$(mktemp -d)
    cd "$TMP_DIR"
    
    # Download and extract
    if command -v curl &> /dev/null; then
        curl -L "https://github.com/shantanusoam/pomo-cli-integrated/archive/main.tar.gz" | tar xz
    elif command -v wget &> /dev/null; then
        wget -qO- "https://github.com/shantanusoam/pomo-cli-integrated/archive/main.tar.gz" | tar xz
    else
        print_error "Neither curl nor wget is available. Please install one of them."
        exit 1
    fi
    
    cd pomo-cli-integrated-main
    
    # Build
    print_status "Building pomo-cli..."
    go mod tidy
    go build -o pomo-cli .
    
    # Install to system
    if [ -w "/usr/local/bin" ]; then
        sudo mv pomo-cli /usr/local/bin/
    else
        print_warning "Cannot install to /usr/local/bin. Installing to ~/.local/bin instead."
        mkdir -p ~/.local/bin
        mv pomo-cli ~/.local/bin/
        
        # Add to PATH if not already there
        if ! echo $PATH | grep -q "$HOME/.local/bin"; then
            echo 'export PATH="$HOME/.local/bin:$PATH"' >> ~/.bashrc
            print_status "Added ~/.local/bin to PATH. Please restart your shell or run: source ~/.bashrc"
        fi
    fi
    
    # Cleanup
    cd / && rm -rf "$TMP_DIR"
    
    print_success "pomo-cli installed successfully!"
}

# Initialize local database
initialize_database() {
    print_status "Initializing local database..."
    
    if command -v pomo-cli &> /dev/null; then
        pomo-cli init
        print_success "Local database initialized!"
    else
        print_error "pomo-cli not found in PATH. Please restart your shell or add it to PATH manually."
    fi
}

# Display usage information
show_usage() {
    echo ""
    print_success "Installation completed! 🍅"
    echo ""
    echo "Quick Start:"
    echo "  pomo-cli add 'Complete project documentation' -p 4 -d 25m -t 'work,docs'"
    echo "  pomo-cli list"
    echo "  pomo-cli start 1"
    echo ""
    echo "For more commands:"
    echo "  pomo-cli --help"
    echo ""
    echo "API Sync (requires pomo-api server):"
    echo "  pomo-cli sync-to-api"
    echo ""
    echo "Local database: ~/.local/share/pomo/pomo.db"
    echo ""
}

# Main installation flow
main() {
    echo "🍅 Pomo CLI Integrated Installation"
    echo "=================================="
    echo ""
    
    print_status "Checking system requirements..."
    check_go
    check_sqlite
    
    print_status "Installing pomo-cli..."
    install_pomo_cli
    
    print_status "Setting up local database..."
    initialize_database
    
    show_usage
}

# Run main function
main "$@"