# WARP.md

This file provides guidance to WARP (warp.dev) when working with code in this repository.

## Tooling and commands

This is a Vite + React + TypeScript single-page application for the knowbuild QERP.CRM marketing site.

- **Install dependencies** (from the repo root):
  - `npm install`
- **Run the development server** (Vite):
  - `npm run dev`
  - The dev server listens on port `3000` and host `0.0.0.0` (see `vite.config.ts`).
- **Build for production**:
  - `npm run build`
- **Preview the production build locally**:
  - `npm run preview`
- **Environment configuration**:
  - Create `.env.local` in the project root and set `GEMINI_API_KEY` to your Gemini API key.
  - Vite injects this value as `process.env.API_KEY` and `process.env.GEMINI_API_KEY` in client code (see `vite.config.ts`).

There are **no test or lint scripts** defined in `package.json` at this time. If you add tooling (e.g. Jest, Vitest, ESLint), also update this file with the relevant commands and how to run individual tests.

## High-level architecture

### Build system and entry points

- **Bundler & dev server**: The app is built and served with **Vite** (see `vite.config.ts`).
  - React support is via `@vitejs/plugin-react`.
  - The `@` alias resolves to the project root (`path.resolve(__dirname, '.')`), so imports like `@/components/...` refer to top-level folders.
- **TypeScript configuration**: `tsconfig.json` targets modern browsers (`ES2022`, `DOM`, `DOM.Iterable`), uses JSX `react-jsx`, and enables `moduleResolution: "bundler"` for Vite.
- **Runtime entry**:
  - `index.tsx` bootstraps React using `ReactDOM.createRoot` on the `#root` element and renders `<App />` inside `React.StrictMode`.

### Application shell and routing

- **App shell**: `App.tsx` defines the global layout and providers:
  - Wraps the application with `HelmetProvider` (for SEO/head tags) and `BrowserRouter` (client-side routing).
  - Applies dark mode by toggling a `dark` class on `document.documentElement`, which is used by the Tailwind-style utility classes.
  - Renders a persistent `<Header />` at the top and `<Footer />` at the bottom around the routed pages.
- **Routing** (React Router v7):
  - Declared in `App.tsx` via `<Routes>` / `<Route>`:
    - `/` → `HomePage`
    - `/solutions` → `SolutionsPage`
    - `/solutions/b2b-qerp-crm` → `SolutionsPage` (SEO-friendly alias for the same page)
    - `/mindset-grow-your-sme` → `MindsetPage`
    - `/questions` → `QuestionsPage`
    - `/contact` → `ContactPage`
    - `/pricing` → `PricingPage`
    - `/privacy-policy` → `PrivacyPolicyPage`
    - Legacy route `/knowbuild/narrative` redirects to `/mindset-grow-your-sme` using `<Navigate />`.
    - A catch-all `*` route redirects unknown paths back to `/`.
  - When adding new pages, follow the existing pattern: create a page component in `pages/` and register it in `App.tsx` with a `<Route>`.

### SEO system

SEO is centralized and driven by configuration:

- **SEO component**: `components/SEO.tsx`
  - Wraps `react-helmet-async`'s `<Helmet>` to set the `<title>`, `meta[name="description"]`, canonical link tags, Open Graph tags, and Twitter card tags.
  - Accepts `structuredData` and injects it as JSON-LD inside a `<script type="application/ld+json">` when provided.
- **SEO configuration**: `utils/seo-config.ts`
  - Exposes a `seoConfig` object with entries for each route (`home`, `solutions`, `mindset`, `questions`, `contact`, `pricing`, `privacyPolicy`, and module-specific entries under `modules`).
  - Also exposes `getCrossLinks(currentPage)` for SEO cross-linking suggestions between related modules.
- **Usage pattern**:
  - Each page component imports `SEO` and `seoConfig` and renders `<SEO />` at the top of the page with props such as `title`, `description`, and `canonical` built from `seoConfig`.
  - Example: `HomePage.tsx` sets SEO by reading from `seoConfig.home`.
  - When adding a new route, first add a corresponding entry in `seoConfig`, then wire it into the page via `SEO`.

### Page composition

Each page in `pages/` is composed from section-level components in `components/`:

- **HomePage** (`pages/HomePage.tsx`):
  - Composes the main marketing story using components like `Hero`, `ProductShowcase`, `RoleStrip`, `Narrative`, `Modules`, `Benefits`, `Trust`, `Pricing`, `FAQ`, and `Contact`.
  - This page sets the pattern for “stacking” themed sections to build out marketing flows.
- **PricingPage** (`pages/PricingPage.tsx`):
  - Uses `SEO` with `seoConfig.pricing` and then renders a dedicated pricing layout with cards, feature lists, and FAQs.
  - Demonstrates more complex, page-specific UI that still leverages shared primitives (e.g., `Button`).
- Other pages (`SolutionsPage`, `MindsetPage`, `QuestionsPage`, `ContactPage`, `PrivacyPolicyPage`) follow a similar structure: start with `SEO`, then compose from section and UI components to build the content.

When introducing new content-heavy pages, prefer composing from existing section components or creating new ones under `components/` rather than embedding large layouts directly in `pages/`.

### Shared UI primitives and animation

- **Section layout primitive**: `components/ui/Section.tsx`
  - Provides a standard section wrapper with configurable `id`, padding, and background (`bgColor: 'white' | 'light'`).
  - Handles vertical spacing (`py-20 md:py-32` by default) and horizontal padding (`px-4 sm:px-6 lg:px-8`).
  - Wraps content in a `motion.div` from `framer-motion` with a fade/slide-in-on-scroll animation and `viewport={{ once: true }}` so sections animate the first time they enter the viewport.
  - Use this to keep layout and scroll-animation behavior consistent across new sections.
- **Buttons and CTAs**: `components/ui/Button.tsx` (not fully documented here, but used pervasively in Header, Pricing, Contact, etc.).
  - Provides size and variant props such as `size="sm" | "lg"`, `variant="primary" | "secondary" | "outline"`, and an optional `fullWidth` prop.
  - Prefer using this component for CTAs instead of raw `<button>` elements to keep styling and behavior consistent.
- **Animations**: `framer-motion` is used for micro-interactions and transitions:
  - `Header.tsx` animates the header into view and handles scroll-based style changes.
  - `Section.tsx` animates sections as they scroll into view.
  - When adding new animated elements, mirror the existing `motion.*` patterns to maintain a coherent animation style.
- **Icons**: `lucide-react` is used for vector icons throughout the UI (e.g., header controls, pricing features, FAQ markers). Import icons from `lucide-react` instead of adding ad hoc SVGs where possible.

### Navigation, theming, and layout behavior

- **Header** (`components/Header.tsx`):
  - Receives `darkMode` and `toggleTheme` props from `App.tsx` and exposes the theme toggle in the header.
  - Uses `useLocation` from React Router to highlight the active nav link based on `location.pathname`.
  - Supports both desktop and mobile navigation:
    - Desktop: inline nav links plus “Book a Demo” / “Contact Sales” buttons.
    - Mobile: hamburger menu that toggles a full-screen overlay menu via `AnimatePresence` from `framer-motion`.
- **Footer** (`components/Footer.tsx`):
  - Provides consistent footer content and likely links across all pages (not detailed here, but used globally in `App.tsx`).
- **Dark mode**:
  - Controlled globally via `darkMode` state in `App.tsx` and toggled on the `<html>` element using the `dark` class.
  - Components rely on Tailwind-style `dark:` variants (e.g., `dark:bg-slate-900`) for theme-specific styling.

### Shared types and data modeling

- **Type definitions**: `types.ts`
  - Centralizes TypeScript interfaces used across UI components:
    - `NavItem` for navigation links.
    - `Feature`, `RoleCardProps`, `Stat`, `FAQItem` for structured marketing and content elements.
  - Import these interfaces in new components rather than redefining similar shapes to keep types aligned.

## How future agents should work in this repo

- Use the commands and architecture notes above to understand how the app is structured before making changes.
- When adding new pages or major sections:
  - Define SEO metadata in `utils/seo-config.ts` and wire it through the `SEO` component.
  - Compose pages from `components/` and `components/ui/` primitives to maintain visual and behavioral consistency.
  - Register new routes in `App.tsx`.
- When integrating new external services or APIs:
  - Prefer environment variables wired through Vite (`loadEnv` + `define`) rather than hard-coding values in the client bundle, consistent with how `GEMINI_API_KEY` is handled.
