'use client';

import React, { useEffect, useState } from 'react';
import Image from 'next/image';
import { Section } from './ui/Section';
import { Play } from 'lucide-react';
import { motion } from 'framer-motion';
import { VideoModal } from './ui/VideoModal';

const MainVideoThumbnail = '/MainVideoThumbnail.png';

export const ProductShowcase: React.FC = () => {
  const [isModalOpen, setIsModalOpen] = useState(false);

  useEffect(() => {
    const openVideo = () => setIsModalOpen(true);

    const onOpenEvent = () => openVideo();
    const onHashChange = () => {
      if (window.location.hash === '#product-showcase') openVideo();
    };

    window.addEventListener('productShowcase:openVideo', onOpenEvent as EventListener);
    window.addEventListener('hashchange', onHashChange);

    // If someone lands directly on this hash, open the modal on mount.
    if (window.location.hash === '#product-showcase') {
      // Give scroll/layout a tick (helps on initial load + smooth scroll)
      window.setTimeout(openVideo, 250);
    }

    return () => {
      window.removeEventListener('productShowcase:openVideo', onOpenEvent as EventListener);
      window.removeEventListener('hashchange', onHashChange);
    };
  }, []);

  return (
    <Section id="product-showcase" className="" theme="solutions">
      <div className="">
        <div className="text-center mb-10">
          <h2 className="text-xl sm:text-2xl md:text-3xl font-bold text-brand-navy dark:text-white">
            See Your Business Operating System in Action
          </h2>
        </div>

        {/* Video Cover Container */}
        <motion.div
          whileHover={{ scale: 1.01 }}
          className="relative group cursor-pointer rounded-2xl md:rounded-[2rem] overflow-hidden border-[18px] border-slate-900/5 dark:border-white/5 shadow-2xl bg-white/50 dark:bg-slate-900/50 backdrop-blur-sm"
          onClick={() => setIsModalOpen(true)}
        >
          {/* Video thumbnail (extracted frame) */}
          <Image
            src={MainVideoThumbnail}
            alt="Knowbuild dashboard overview – business operating system in action"
            width={1920}
            height={1080}
            className="w-full h-auto object-cover"
            sizes="(max-width: 768px) 100vw, (max-width: 1200px) 80vw, 1200px"
            priority
          />

          {/* Overlay Gradient */}
          <div className="absolute inset-0 bg-brand-navy/10 group-hover:bg-brand-navy/20 dark:bg-black/20 dark:group-hover:bg-black/30 transition-colors duration-300" />
          <div className="absolute left-1/2 -translate-x-1/2 bottom-8 w-[92%] sm:w-[80%] md:w-[70%] px-4 py-3 rounded-lg bg-black/70 dark:bg-white/10 text-white dark:text-white text-center text-lg sm:text-xl md:text-2xl font-semibold shadow-2xl pointer-events-none select-none"
                style={{
                  textShadow: '0 4px 16px rgba(0,0,0,0.55), 0 1px 0px #222',
                  lineHeight: 1.4,
                  letterSpacing: 0.01,
                }}
              >
              Built and used inside <span className="text-brand-orange font-bold">fast-growing SMEs</span> — delivering <span className="text-brand-orange font-bold">3× growth</span> every 3–4 years with clarity, speed &amp; control.
              <br />
              <span className="block text-white mt-2" >
                A <span className="text-brand-orange font-bold">2-minute</span>  walkthrough that shows how knowbuild connects all business functions—into one powerful system.
              </span>
            </div>
          {/* Play Button */}
          <div className="absolute inset-0 flex items-center justify-center">
            
            <motion.div
              initial={{ scale: 0.9 }}
              whileHover={{ scale: 1.1 }}
              className="w-20 h-20 md:w-24 md:h-24 bg-white/90 backdrop-blur-xl rounded-full flex items-center justify-center shadow-glow-orange group-hover:shadow-[0_0_50px_rgba(254,108,0,0.6)] transition-all duration-300"
            >
              <Play className="w-8 h-8 md:w-10 md:h-10 text-brand-orange ml-1.5 fill-brand-orange" />
          
            </motion.div>
          </div>

          {/* Pulse Effect behind play button */}
          <div className="absolute top-1/2 left-1/2 -translate-x-1/2 -translate-y-1/2 w-20 h-20 bg-white/30 rounded-full animate-ping pointer-events-none" />
        </motion.div>
      </div>

      <VideoModal
        isOpen={isModalOpen}
        onClose={() => setIsModalOpen(false)}
        videoUrl="https://knowbuildwebsiteassets.s3.ap-south-1.amazonaws.com/knowbuild-homepage-cta-web.mp4"
        title="knowbuild Platform Overview"
        autoPlay
      />
    </Section>
  );
};
