'use client';

import React, { useRef, useState } from 'react';
import Image from 'next/image';
import { motion, useScroll, useTransform, useSpring, useMotionValue, useMotionTemplate, AnimatePresence, LayoutGroup } from 'framer-motion';
import {
  ArrowRight, CheckCircle2, LayoutGrid, BarChart3, Users,
  ShieldCheck, Zap, MessageSquare, FileText, Truck,
  ShoppingCart, Receipt, Layers, TrendingUp, Bell, Search, Mail, Smartphone, Play,
  Wrench, Package, Gavel, ChevronDown, LucideIcon, X
} from 'lucide-react';
import { Button } from './ui/Button';
import { useSectionTheme } from '../hooks/useSectionTheme';
import GradientBackground from './animations/GradientBackground';
import { DashboardCarousel } from './DashboardCarousel';

// --- Animation & UI Components ---

const FadeIn = ({ children, delay = 0, className = "" }: { children: React.ReactNode, delay?: number, className?: string }) => (
  <motion.div
    initial={{ opacity: 0, y: 30, filter: "blur(8px)" }}
    whileInView={{ opacity: 1, y: 0, filter: "blur(0px)" }}
    viewport={{ once: true, margin: "-50px" }}
    transition={{ duration: 0.8, delay, ease: [0.21, 0.47, 0.32, 0.98] }}
    className={className}
  >
    {children}
  </motion.div>
);

const RevealText = ({ text, className = "" }: { text: string, className?: string }) => {
  const words = text.split(" ");
  return (
    <span className={`inline-flex flex-wrap gap-x-3 gap-y-1 ${className}`}>
      {words.map((word, i) => (
        <span key={i} className="overflow-hidden inline-block">
          <motion.span
            initial={{ y: "100%" }}
            whileInView={{ y: 0 }}
            viewport={{ once: true }}
            transition={{ duration: 0.5, delay: i * 0.03, ease: "backOut" }}
            className="inline-block"
          >
            {word}
          </motion.span>
        </span>
      ))}
    </span>
  );
};

type SpotlightCardProps = Omit<React.ComponentPropsWithoutRef<typeof motion.div>, 'children'> & {
  children: React.ReactNode;
  delay?: number;
};

const SpotlightCard = ({ children, className = "", delay = 0, onClick, layoutId, ...props }: SpotlightCardProps) => {
  const mouseX = useMotionValue(0);
  const mouseY = useMotionValue(0);

  function handleMouseMove({ currentTarget, clientX, clientY }: React.MouseEvent) {
    const { left, top } = currentTarget.getBoundingClientRect();
    mouseX.set(clientX - left);
    mouseY.set(clientY - top);
  }

  return (
    <motion.div
      layout
      layoutId={layoutId}
      initial={{ opacity: 0, scale: 0.95 }}
      whileInView={{ opacity: 1, scale: 1 }}
      viewport={{ once: true }}
      transition={{ duration: 0.5, delay, layout: { duration: 0.3, ease: "easeOut" } }}
      onMouseMove={handleMouseMove}
      onClick={onClick}
      className={`group relative overflow-hidden rounded-3xl border border-slate-200 dark:border-white/10 bg-white dark:bg-slate-800/50 shadow-lg dark:shadow-none ${onClick ? 'cursor-pointer' : ''} ${className}`}
      {...props}
    >
      <motion.div
        className="pointer-events-none absolute -inset-px rounded-3xl opacity-0 transition duration-300 group-hover:opacity-100"
        style={{
          background: useMotionTemplate`
            radial-gradient(
              650px circle at ${mouseX}px ${mouseY}px,
              rgba(14, 165, 233, 0.1),
              transparent 80%
            )
          `,
        }}
      />
      <div className="relative h-full p-8">{children}</div>
    </motion.div>
  );
};

// --- Module Data & Card Component ---

interface ModuleData {
  title: string;
  desc: string;
  icon: LucideIcon;
  accent: string;
  accentBg: string;
  features: string[];
  isOptional?: boolean;
}

export const modulesData: ModuleData[] = [
  {
    title: "Sales",
    desc: "Inquiry to order closure sales workflow, linked with dispatch & accounts receivable.",
    icon: TrendingUp,
    accent: "text-blue-500",
    accentBg: "bg-blue-500/10",
    features: [
      "Inquiry to order closure sales workflow, linked with dispatch & accounts receivable.",
      "Automated reminders or missed inquiries and quotation follow-ups.",
      "Sales task management including meeting scheduling and visit planning with reminders.",
      "Sales target tracking by product and region using PMS (Product Market Sheet).",
      "Macro and micro target monitoring for annual and monthly goals.",
      "Proforma invoice and delivery order integration for error-free order processing.",
      "Sales funnel visualization and conversion rate analysis.",
      "Customer retargeting.",
      "Archives of all offers generated, proforma invoices, purchase order copies perpetual basis."
    ]
  },
  {
    title: "Inbound Sales",
    desc: "Capture and assign every inquiry from calls, emails, or web. Route leads instantly.",
    icon: MessageSquare,
    accent: "text-indigo-500",
    accentBg: "bg-indigo-500/10",
    features: [
      "Seamless website API integration to capture inquiries directly from your website.",
      "Inquiry assignment based on predefined business rules.",
      "Automated Escalation alerts for unassigned inquiries to ensure no inquiry is missed.",
      "Inquiry trend Analysis and ROI Tracking of inquiry Platforms like Google* and IndiaMart*."
    ]
  },
  {
    title: "Finance",
    desc: "One-click digital invoicing connected to accounts receivable. Get faster billing and cleaner ageing reports.",
    icon: Receipt,
    accent: "text-orange-500",
    accentBg: "bg-orange-500/10",
    features: [
      "E-Invoicing integration via GST portal and E-Way bill generation.",
      "Customer credit limit management with automated revocation on delayed payments.",
      "Dynamic forex integration for import/export transactions.",
      "Accounts receivable aging reports with follow-up logs and reminders.",
      "Cash flow forecasting based on customer payment behavior and vendor commitments.",
      "Accounts payable heatmaps, ledgers, and credit note management."
    ]
  },
  {
    title: "Procurement",
    desc: "Automated purchase recommendations based on sales forecasts. Manage vendors smarter.",
    icon: ShoppingCart,
    accent: "text-purple-500",
    accentBg: "bg-purple-500/10",
    features: [
      "Vendor management system with keyword based filtering.",
      "Vendor-product linkage using unique part codes.",
      "Inventory forecasting based on sales funnel data.",
      "Automated stock breach alerts to guide purchasing decisions.",
      "Purchase trend analytics by month, quarter, year and vendor.",
      "Price comparatives for current vs. previous year purchases.",
      "Integrated purchasing warehousing & accounting for payment scheduling."
    ]
  },
  {
    title: "Warehousing",
    desc: "Inventory visibility and multi-location stock management. Gain better control across branches.",
    icon: Package,
    accent: "text-pink-500",
    accentBg: "bg-pink-500/10",
    features: [
      "Integrated Inventory & Logistics Management End-to-end warehouse control Auto-generated serial numbers for tracking.",
      "Incoming Manager Real-time logging of received stock with Immediate inventory updates for accuracy.",
      "Accept incoming orders as partial or complete integrating with finance module for vendor payment.",
      "Return Credit Notes: Material returned can be restocked with credit notes generation.",
      "Demo Stock Manager Tracks trial/sample stock Separate from main inventory.",
      "Inventory Manager Dashboard Centralized view of stock & movement - Aids smart decisions & resource used.",
      "Stock Reports Outgoing: Track dispatched goods."
    ]
  },
  {
    title: "Dispatch",
    desc: "Error-free, streamlined dispatch linked directly to sales orders. Reduce wrong shipments drastically.",
    icon: Truck,
    accent: "text-emerald-500",
    accentBg: "bg-emerald-500/10",
    features: [
      "Sales-generated delivery orders based on UPC's provide dispatch teams ready to execute instructions directly on the dispatch dashboard - no manual instruction required.",
      "Unique part codes enables zero-error dispatch execution.",
      "Ensures 100% accuracy through, Barcode scanning Smart pick-pack workflows & Real-time inventory validation.",
      "Integrated with finance for: Compliance checks, approval verification.",
      "Linked with logistics for optimized delivery routing.",
      "Connectivity for e-invoicing e-way bill with GST portal.",
      "Create dispatch credit notes for invoice reversal performance analytics for dispatch efficiency.",
      "Makes dispatch seamless, efficient, and reliable."
    ]
  },
  {
    title: "Tendering",
    desc: "Seamless coordination for government and private tenders. Track deadlines and reverse auctions.",
    icon: Gavel,
    accent: "text-cyan-500",
    accentBg: "bg-cyan-500/10",
    features: [
      "Manages complete tender lifecycle from creation to result.",
      "Create and upload tenders with detailed info & documents.",
      "Set due dates with automated reminders.",
      "Collaborative tools for team coordination & transparency.",
      "Assign tenders based on team expertise.",
      "Centralized dashboard with real-time tender status.",
      "Delivers full visibility, automation, and accountability.",
      "Logs results with internal approval workflows.",
      "Structured, professional bid management approach."
    ],
    isOptional: true
  },
  {
    title: "Service Support",
    desc: "Manage service requests for warranty and out-of-warranty products with complete tracking.",
    icon: Wrench,
    accent: "text-amber-500",
    accentBg: "bg-amber-500/10",
    features: [
      "Warranty Support Manages service requests for in-warranty products, tracks warranty period, terms and coverage.",
      "Out of Warranty Handles service for products with expired warranty. And generates paid service estimates & approvals, tracks parts used and billing details."
    ],
    isOptional: true
  }
];

interface ModuleCardProps {
  module: ModuleData;
  index: number;
  onToggle: () => void;
}

const ModuleCard: React.FC<ModuleCardProps> = ({ module, index, onToggle }) => {
  const Icon = module.icon;

  return (
    <SpotlightCard
      layoutId={`card-${module.title}`}
      delay={index * 0.05}
      onClick={onToggle}
      className="flex flex-col h-full cursor-pointer"
    >

      <motion.div layout="position" className="flex-shrink-0">
        {/* Header */}
        <div className="flex items-start justify-between mb-4">
          <motion.div layoutId={`icon-${module.title}`} className={`w-12 h-12 rounded-2xl ${module.accentBg} flex items-center justify-center border border-slate-200 dark:border-white/10 ${module.accent}`}>
            <Icon size={24} />
          </motion.div>

          <motion.div layoutId={`chevron-${module.title}`} className="text-slate-400 dark:text-slate-500">
            <ChevronDown size={20} className="-rotate-90" />
          </motion.div>
        </div>

        <motion.h3 layoutId={`title-${module.title}`} className="text-xl font-bold mb-2 text-brand-navy dark:text-white">{module.title}    {module.isOptional && (
          <motion.span layout="position" className="mt-auto pt-4">
            <span className={`text-xs font-semibold ${module.accent} opacity-70`}>
              Optional
            </span>
          </motion.span>
        )}</motion.h3>
        <motion.p layoutId={`desc-${module.title}`} className="text-slate-600 dark:text-slate-400 leading-relaxed text-sm font-medium line-clamp-3">{module.desc}</motion.p>
      </motion.div>

      {/* Footer hint */}
      <motion.div layout="position" className="mt-auto pt-4">
        <span className={`text-xs font-semibold ${module.accent} opacity-70`}>
          Click to view features
        </span>
      </motion.div>
    </SpotlightCard>
  );
};

const ExpandedModuleOverlay = ({ module, onClose }: { module: ModuleData, onClose: () => void }) => {
  const Icon = module.icon;

  // Handle click outside
  React.useEffect(() => {
    const handleKeyDown = (e: KeyboardEvent) => {
      if (e.key === 'Escape') onClose();
    };
    window.addEventListener('keydown', handleKeyDown);
    return () => window.removeEventListener('keydown', handleKeyDown);
  }, [onClose]);

  return (
    <div className="fixed inset-0 z-50 flex items-center justify-center px-4 sm:px-6">
      <motion.div
        initial={{ opacity: 0 }}
        animate={{ opacity: 1 }}
        exit={{ opacity: 0 }}
        onClick={onClose}
        className="absolute inset-0 bg-black/60 backdrop-blur-sm"
      />
      <motion.div
        layoutId={`card-${module.title}`}
        className="relative w-full max-w-2xl bg-white dark:bg-[#131b2c] rounded-3xl shadow-2xl overflow-hidden z-10 border border-slate-200 dark:border-white/10 max-h-[600px] md:max-h-[700px] lg:max-h-[85vh] flex flex-col"
      >
        {/* Close Button */}
        <button
          onClick={(e) => { e.stopPropagation(); onClose(); }}
          className="absolute top-4 right-4 p-2 bg-slate-100 dark:bg-white/10 rounded-full text-slate-500 hover:text-brand-navy dark:hover:text-white transition-colors z-20"
        >
          <X size={20} />
        </button>

        <div className="p-8 md:p-10 overflow-y-auto custom-scrollbar">
          <div className="flex items-start gap-6 mb-8">
            <motion.div layoutId={`icon-${module.title}`} className={`w-16 h-16 rounded-2xl ${module.accentBg} flex-shrink-0 flex items-center justify-center border border-slate-200 dark:border-white/10 ${module.accent}`}>
              <Icon size={32} />
            </motion.div>
            <div>
              <motion.h3 layoutId={`title-${module.title}`} className="text-3xl font-bold mb-3 text-brand-navy dark:text-white">
                {module.title}
              </motion.h3>
              <motion.p layoutId={`desc-${module.title}`} className="text-slate-600 dark:text-slate-400 text-lg leading-relaxed">
                {module.desc}
              </motion.p>
            </div>
          </div>

          <motion.div
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ delay: 0.2 }}
            className="border-t border-slate-100 dark:border-white/10 pt-8"
          >
            <h4 className={`text-sm font-bold uppercase tracking-wider mb-6 ${module.accent}`}>
              Key Features
            </h4>
            <div className="grid md:grid-cols-2 gap-4">
              {module.features.map((feature, i) => (
                <motion.div
                  key={i}
                  initial={{ opacity: 0, x: -10 }}
                  animate={{ opacity: 1, x: 0 }}
                  transition={{ delay: 0.3 + (i * 0.05) }}
                  className="flex items-start gap-3 text-slate-600 dark:text-slate-400"
                >
                  <CheckCircle2 className={`flex-shrink-0 mt-1 ${module.accent}`} size={16} />
                  <span className="text-sm leading-relaxed font-medium">{feature}</span>
                </motion.div>
              ))}
            </div>
          </motion.div>
        </div>
      </motion.div>
    </div>
  );
};

// --- Sections ---

export const Solutions: React.FC = () => {
  const containerRef = useRef(null);
  const themeRef = useSectionTheme('solutions');
  const { scrollYProgress } = useScroll({ target: containerRef });
  const yHero = useTransform(scrollYProgress, [0, 1], [0, -100]);
  const [openModuleIndex, setOpenModuleIndex] = useState<number | null>(null);

  const handleModuleToggle = (index: number) => {
    setOpenModuleIndex(prev => prev === index ? null : index);
  };

  return (
    <div ref={containerRef} className="bg-transparent text-slate-900 dark:text-white min-h-screen font-sans selection:bg-brand-orange/30 selection:text-brand-orange pb-20 transition-colors duration-500">

      {/* 1. HERO – CRM.Q-ERP Solutions Overview */}
      <section ref={themeRef} className="relative min-h-[600px] md:min-h-[700px] lg:min-h-[95vh] flex items-center justify-center pt-24 md:pt-32 px-4 sm:px-6 lg:px-8 overflow-hidden">
        {/* Ambient Background */}
        <div className="absolute top-[-20%] right-[-10%] w-[800px] h-[800px] bg-brand-blue/20 dark:bg-brand-blue/10 rounded-full blur-[120px] mix-blend-multiply dark:mix-blend-screen animate-float" />
        <div className="absolute bottom-[-20%] left-[-10%] w-[600px] h-[600px] bg-brand-orange/20 dark:bg-brand-orange/10 rounded-full blur-[100px] mix-blend-multiply dark:mix-blend-screen animate-float" style={{ animationDelay: '2s' }} />

        <div className="max-w-7xl mx-auto w-full relative z-10 text-center md:text-left">
          <FadeIn>
            <span className="inline-flex items-center gap-2 py-1 px-3 rounded-full border border-brand-blue/20 bg-brand-blue/5 text-brand-blue text-xs font-bold uppercase tracking-widest mb-8 backdrop-blur-md">
              <span className="w-2 h-2 rounded-full bg-brand-blue animate-pulse" />
              Advanced Q-ERP.CRM for B2B SMEs
            </span>
          </FadeIn>

          <div className="mb-1">
            <h1 className="text-4xl md:text-7xl font-bold leading-[1] tracking-tighter text-brand-navy dark:text-white mb-4">
              <RevealText text="Automate. Unify." /> <br className="hidden md:block" />
              <span className="text-brand-orange bg-clip-text bg-gradient-to-r from-brand-blue to-brand-teal">
                <RevealText text="Systemize." />
              </span>
            </h1>
            <p className="text-slate-600 dark:text-slate-400 mb-1 leading-relaxed text-2xl md:text-3xl">All your business operations.</p>
          </div>

          <div className="grid lg:grid-cols-2 gap-12 items-center">
            <FadeIn delay={0.2} className="max-w-2xl">
              <h2 className="text-lg md:text-xl text-slate-700 dark:text-slate-200 font-medium mb-3 leading-tight">
                A single operating system that unifies CRM and ERP.
                From inquiry to invoice, run your B2B SME on one system.
              </h2>
              <p className="text-slate-600 dark:text-slate-400 text-lg mb-10 leading-relaxed">
                Built for growing SMEs.
                Knowbuild replaces fragmented tools with an integrated QERP.CRM, providing real-time, reliable data so you can take clear, confident decisions — and scale with control.
              </p>

              <div className="flex flex-col sm:flex-row gap-4 justify-center md:justify-start">
                <a href="/book-demo">
                  <Button variant="primary" size="lg" className="shadow-glow-orange shadow-brand-orange/20 w-full sm:w-auto">
                    Book a Demo
                  </Button>
                </a>
                <a href="/contact">
                  <Button variant="outline" size="lg" className="border-slate-300 dark:border-white/20 hover:bg-slate-100 dark:hover:bg-white/10 text-slate-900 dark:text-white w-full sm:w-auto">
                    Contact Sales
                  </Button>
                </a>
              </div>
            </FadeIn>

            {/* Dynamic Dashboard Carousel */}
            <DashboardCarousel yHero={yHero} />
          </div>
        </div>
      </section>

      {/* 2. "GET TO KNOWBUILD" – QERP.CRM BUSINESS MODULES */}
      <section id="business-modules" className="py-24 px-4 sm:px-6 lg:px-8 relative">
        <div className="max-w-[1400px] mx-auto">
          <div className="mb-16">
            <FadeIn>
              <span className="text-brand-blue font-bold tracking-widest text-sm uppercase mb-2 block">Get to Know the Platform</span>
              <h2 className="text-4xl md:text-6xl font-bold mb-6 text-brand-navy dark:text-white tracking-tight">Q-ERP.CRM Business Modules</h2>
              <p className="text-xl text-slate-600 dark:text-slate-400 max-w-3xl leading-relaxed">
                knowbuild is a unified system integrating key business functions such as Sales, Inbound Sales, Dispatch, Finance, Procurement, Warehousing, Tendering, and Service Support. We ensure seamless data flow and zero silos.
              </p>
            </FadeIn>
          </div>

          {/* 3. MODULE CARDS – 4-COLUMN GRID */}
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-5 items-start">
            {modulesData.map((module, i) => (
              <ModuleCard
                key={module.title}
                module={module}
                index={i}
                onToggle={() => handleModuleToggle(i)}
              />
            ))}
          </div>

          {/* Expanded Overlay */}
          <AnimatePresence>
            {openModuleIndex !== null && (
              <ExpandedModuleOverlay
                module={modulesData[openModuleIndex]}
                onClose={() => setOpenModuleIndex(null)}
              />
            )}
          </AnimatePresence>

          {/* Disclaimer */}
          {/* <FadeIn delay={0.5}>
            <p className="text-xs text-slate-400 dark:text-slate-500 mt-10 text-center max-w-2xl mx-auto">
              <span className="font-semibold">Disclaimer:</span> All product and company names (e.g., Excel, IndiaMart, Google) are trademarks of their respective owners, with no affiliation or endorsement implied.
            </p>
          </FadeIn> */}
        </div>
      </section>

      {/* 4. “THE HEART OF ANY BUSINESS” – MANAGEMENT FOCUS */}
      <section className="py-32 relative overflow-hidden bg-white/50 dark:bg-black/40 backdrop-blur-sm transition-colors duration-500">
        <div className="absolute inset-0 bg-[linear-gradient(to_right,#80808012_1px,transparent_1px),linear-gradient(to_bottom,#80808012_1px,transparent_1px)] bg-[size:24px_24px]" />

        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 relative z-10">
          <FadeIn>
            <span className="text-brand-orange text-xs font-bold tracking-widest uppercase mb-4 block">The Heart of Any Business</span>
            <h2 className="text-4xl md:text-5xl font-bold mb-16 text-brand-navy dark:text-white tracking-tight">Management Visibility</h2>
          </FadeIn>

          <div className="grid md:grid-cols-3 gap-12">
            {[
              {
                title: "Sales Management",
                desc: "Democratize the sales process. Enforce organizational rules enabling sales teams to sell 2 ~ 3 x faster,and access analytics so leaders can assess team, region, and product performance objectively in Real Time",
                icon: Users
              },
              {
                title: "Finance & Finance",
                desc: "Gain visibility into receivables, debtor ageing, credit limits, and cash-flow. Finance can finally control risk while enabling the sales team to grow.",
                icon: BarChart3
              },
              {
                title: "Purchase Management",
                desc: "Plan purchases based on data, not gut feel. Align buying decisions to sales demand and inventory realities to optimize working capital.",
                icon: ShoppingCart
              }
            ].map((item, i) => (
              <FadeIn key={i} delay={i * 0.2} className="relative pl-8 border-l-2 border-slate-100 dark:border-white/10 hover:border-brand-blue transition-colors duration-300 group">
                <div className="mb-6 text-brand-blue group-hover:scale-110 transition-transform duration-300 origin-left">
                  <item.icon size={32} />
                </div>
                <h3 className="text-2xl font-bold mb-4 text-brand-navy dark:text-white">{item.title}</h3>
                <p className="text-slate-600 dark:text-slate-400 leading-relaxed">{item.desc}</p>
              </FadeIn>
            ))}
          </div>
        </div>
      </section>

      {/* 5. “OVERVIEW” – MANAGEMENT DASHBOARD NARRATIVE */}
      <section className="py-32 px-4 sm:px-6 lg:px-8 bg-slate-50/50 dark:bg-slate-900/40 backdrop-blur-sm transition-colors duration-500">
        <div className="max-w-7xl mx-auto">
          <div className="grid lg:grid-cols-2 gap-20 items-center">
            <FadeIn>
              <h3 className="text-3xl font-bold mb-6 flex items-center gap-3 text-brand-navy dark:text-white">
                <LayoutGrid className="text-brand-teal" /> Overview
              </h3>
              <p className="text-xl text-slate-600 dark:text-slate-300 leading-relaxed mb-8">
                Imagine a dashboard that tells the whole truth. Revenue achieved vs targets, account receivables, orders in hand, and live opportunities—all in one view.
              </p>
              <p className="text-lg text-slate-500 leading-relaxed mb-6">
                A dedicated "Today's Tasks" area ensures visits, follow-ups, and reporting tasks are never missed. Management can finally see pipeline health, performance gaps, and financial risk at a single glance.
              </p>
            </FadeIn>

            <motion.div
              initial={{ opacity: 0, x: 50 }}
              whileInView={{ opacity: 1, x: 0 }}
              transition={{ duration: 0.8, delay: 0.2 }}
              className="relative"
            >
              <div className="absolute inset-0 bg-brand-blue/20 blur-3xl -z-10 rounded-full opacity-50" />
              <div className="bg-white dark:bg-[#131b2c] rounded-2xl border border-slate-200 dark:border-white/10 shadow-2xl overflow-hidden">
                <div className="dark:hidden">
                  <Image
                    src="/TargetSummaryDashbaord.png"
                    alt="Target Summary Dashboard"
                    width={1200}
                    height={800}
                    className="w-full h-auto rounded-2xl"
                    priority
                  />
                </div>
                <div className="hidden dark:block">
                  <Image
                    src="/TargetSummaryDashbaord.png"
                    alt="Target Summary Dashboard"
                    width={1200}
                    height={800}
                    className="w-full h-auto rounded-2xl brightness-90 contrast-110"
                    priority
                  />
                </div>
              </div>
            </motion.div>
          </div>
        </div>
      </section>

      {/* 6. “SIMPLY TRANSPARENCY” – BOOST EFFICIENCY */}
      <section className="py-24 bg-white dark:bg-white/5 backdrop-blur-sm border-y border-slate-200 dark:border-white/5 transition-colors duration-500">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <FadeIn className="text-center mb-16">
            <span className="text-brand-teal font-mono text-sm mb-2 block">Simply Transparency</span>
            <h2 className="text-5xl font-bold text-brand-navy dark:text-white">Boost Efficiency</h2>
          </FadeIn>

          <div className="grid md:grid-cols-2 gap-10">
            <SpotlightCard className="group">
              <div className="flex items-center gap-4 mb-6">
                <div className="p-3 rounded-xl bg-brand-blue/10 text-brand-blue"><CheckCircle2 size={32} /></div>
                <h3 className="text-2xl font-bold text-brand-navy dark:text-white">Task Manager</h3>
              </div>
              <p className="text-slate-600 dark:text-slate-400 mb-6 leading-relaxed">
                knowbuild replaces Excel sheets with a centralized task manager. If an enquiry follow-up or offer revision is missed, the system alerts via email/WhatsApp and escalates automatically.
              </p>
              <div className="bg-slate-50 dark:bg-white/5 p-4 rounded-lg border border-slate-100 dark:border-white/5">
                <p className="text-slate-500 text-sm font-medium">
                  <span className="text-brand-blue font-bold">Outcome:</span> Fewer missed follow-ups, automated reporting.
                </p>
              </div>
            </SpotlightCard>

            <SpotlightCard className="group">
              <div className="flex items-center gap-4 mb-6">
                <div className="p-3 rounded-xl bg-brand-orange/10 text-brand-orange"><Search size={32} /></div>
                <h3 className="text-2xl font-bold text-brand-navy dark:text-white">Knowledge Center (Optional)</h3>
              </div>
              <p className="text-slate-600 dark:text-slate-400 mb-6 leading-relaxed">
                A centralized knowledge bank for standard documents, processes, and reference material. Reduce duplicate work and help new employees ramp up faster.
              </p>
              <div className="bg-slate-50 dark:bg-white/5 p-4 rounded-lg border border-slate-100 dark:border-white/5">
                <p className="text-slate-500 text-sm font-medium">
                  <span className="text-brand-orange font-bold">Outcome:</span> Transparent knowledge sharing alongside operational systems.
                </p>
              </div>
            </SpotlightCard>
          </div>
        </div>
      </section>

      {/* 7. “KEY RESULT” – CULTURE & IMPACT */}
      <section className="py-32 px-4 relative overflow-hidden bg-brand-navy dark:bg-[#050910] text-white">
        <div className="absolute inset-0">
          <GradientBackground />
        </div>


        <div className="absolute inset-0 bg-[url('https://grainy-gradients.vercel.app/noise.svg')] opacity-20 mix-blend-overlay" />
        <div className="absolute top-0 right-0 w-[600px] h-[600px] bg-brand-teal/20 rounded-full blur-[100px]" />

        <div className="max-w-5xl mx-auto text-center relative z-10">
          <FadeIn>
            <span className="text-white/50 uppercase tracking-[0.2em] text-sm mb-6 block">Key Result</span>
            <h2 className="text-4xl md:text-6xl font-bold leading-tight mb-8">
              Turn your company into a <br />
              <span className="text-transparent bg-clip-text bg-gradient-to-r from-brand-teal to-white">High-Performance Culture</span>.
            </h2>
            <p className="text-xl text-slate-300 leading-relaxed max-w-3xl mx-auto">
              knowbuild is more than just software. It transforms your culture by eliminating repetitive, mundane tasks, allowing teams to focus on strategic thinking and growth. The system embeds discipline, transparency, and standardized processes into everyday work.
            </p>
          </FadeIn>
        </div>
      </section>

      {/* 8. METRICS STRIP */}
      <div className="border-y border-slate-200 dark:border-white/10 bg-slate-50 dark:bg-[#0B1220] transition-colors duration-500">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
          <div className="grid md:grid-cols-3 gap-12 divide-y md:divide-y-0 md:divide-x divide-slate-200 dark:divide-white/10">
            {[
              { label: "Grow Sales Productivity", value: "37%", suffix: "Up to", desc: "Structured processes help teams close more deals with less effort." },
              { label: "Optimize Workflow", value: "50%", suffix: "Up to", desc: "Better coordination between departments through automation." },
              { label: "Unify & Systemize", value: "1", suffix: "System", desc: "One unified platform reduces errors, duplication, and miscommunication." },
            ].map((metric, i) => (
              <div key={i} className="text-center px-4 pt-8 md:pt-0">
                <p className="text-brand-orange font-bold text-sm mb-2 tracking-wider uppercase">{metric.suffix}</p>
                <div className="text-6xl font-bold text-brand-navy dark:text-white mb-4">{metric.value}</div>
                <h4 className="text-lg font-bold text-brand-navy dark:text-white mb-2">{metric.label}</h4>
                <p className="text-slate-600 dark:text-slate-400 text-sm">{metric.desc}</p>
              </div>
            ))}
          </div>
        </div>
      </div>

      {/* 9. INTEGRATIONS */}
      <section className="py-24 px-4 bg-white/50 dark:bg-black/40 backdrop-blur-sm transition-colors duration-500">
        <div className="max-w-4xl mx-auto text-center mb-16">
          <FadeIn>
            <span className="text-brand-blue font-bold uppercase text-xs mb-4 block">Integrations</span>
            <h2 className="text-4xl font-bold mb-6 text-brand-navy dark:text-white">Remove Noise, Get Instant Notifications</h2>
            <p className="text-slate-600 dark:text-slate-400">
              knowbuild connects with key channels like Email, WhatsApp, and SMS to send actionable alerts without overwhelming users.
            </p>
          </FadeIn>
        </div>

        <div className="max-w-5xl mx-auto grid md:grid-cols-3 gap-6">
          {[
            { title: "Email", icon: Mail, desc: "Receive daily digests and task escalation summaries directly in your inbox." },
            { title: "WhatsApp", icon: MessageSquare, desc: "Critical updates reach teams where they already are, enabling fast action." },
            { title: "In-App", icon: Bell, desc: "Quick, contextual notifications for offers, payments, tasks, and approvals." }
          ].map((card, i) => (
            <SpotlightCard key={i} delay={i * 0.1} className="text-left">
              <div className="mb-4 w-12 h-12 rounded-2xl bg-slate-100 dark:bg-white/10 flex items-center justify-center text-brand-navy dark:text-white">
                <card.icon size={24} />
              </div>
              <h3 className="text-xl font-bold mb-2 text-brand-navy dark:text-white">{card.title}</h3>
              <p className="text-slate-600 dark:text-slate-400 text-sm leading-relaxed">{card.desc}</p>
            </SpotlightCard>
          ))}
        </div>
      </section>

      {/* 10. FAQ SECTION */}
      <section className="py-24 bg-slate-50/50 dark:bg-slate-900/40 backdrop-blur-sm transition-colors duration-500">
        <div className="max-w-3xl mx-auto px-4">
          <h2 className="text-3xl font-bold mb-12 text-center text-brand-navy dark:text-white">Frequently Asked Questions</h2>
          <div className="space-y-4">
            {[
              { q: "What exactly is knowbuild Q-ERP.CRM?", a: "Unlike standalone tools, knowbuild is a unified operating system purpose-built for B2B SMEs. It connects the entire inquiry-to-cash cycle, linking sales directly to operations without enterprise complexity." },
              { q: "Is it suitable for multiple locations?", a: "Absolutely. knowbuild is cloud-native and designed to manage stock, sales, and operations across multiple branches and cities seamlessly." },
              { q: "How long does it take to go live?", a: "For a typical SME, we aim for a 2-4 week Go-Live. Our team assists with data migration so you can start quoting and billing quickly." },
              { q: "Can we start with just a few modules?", a: "Yes. You can roll out in phases—start by fixing your sales funnel, then unlock procurement and dispatch when your team is ready." },
              { q: "How secure is our data?", a: "We use bank-level encryption and strict access controls. Your business data is isolated and protected with enterprise-grade security standards." }
            ].map((faq, i) => (
              <details key={i} className="group bg-white dark:bg-white/5 border border-slate-200 dark:border-white/10 rounded-2xl overflow-hidden transition-all duration-300 open:shadow-lg">
                <summary className="flex justify-between items-center p-6 cursor-pointer list-none font-medium text-lg text-brand-navy dark:text-slate-200 hover:text-brand-blue dark:hover:text-white transition-colors">
                  {faq.q}
                  <span className="transition-transform group-open:rotate-180 text-slate-400 ml-4">↓</span>
                </summary>
                <div className="px-6 pb-6 text-slate-600 dark:text-slate-400 leading-relaxed bg-slate-50/50 dark:bg-white/0">
                  {faq.a}
                </div>
              </details>
            ))}
          </div>
        </div>
      </section>

      {/* 11. PRICING & 12. ONBOARDING */}
      <section className="py-24 px-4 relative overflow-hidden">
        <div className="max-w-5xl mx-auto bg-white dark:bg-[#131b2c] rounded-[3rem] shadow-2xl border border-slate-200 dark:border-white/10 overflow-hidden">
          <div className="grid md:grid-cols-2">
            <div className="p-12 md:p-16 flex flex-col justify-center">
              <span className="text-brand-teal font-bold uppercase text-xs mb-4 block">Affordable, Powerful Software</span>
              <h2 className="text-3xl md:text-4xl font-bold mb-6 text-brand-navy dark:text-white">Priced for Growth. <br />Built for You.</h2>
              <p className="text-slate-600 dark:text-slate-400 mb-6 leading-relaxed">
                knowbuild is priced with B2B SMEs in mind. We believe in the philosophy: "We grow when you grow." No expensive consultants needed.
              </p>
              <div className="flex flex-col gap-4 mt-4">
                <div className="flex gap-3 items-start">
                  <CheckCircle2 className="text-brand-teal flex-shrink-0" />
                  <div>
                    <h4 className="font-bold text-brand-navy dark:text-white text-sm">Fast Onboarding</h4>
                    <p className="text-xs text-slate-500">2-4 week typical go-live with assisted data migration.</p>
                  </div>
                </div>
                <div className="flex gap-3 items-start">
                  <CheckCircle2 className="text-brand-teal flex-shrink-0" />
                  <div>
                    <h4 className="font-bold text-brand-navy dark:text-white text-sm">No Hidden Costs</h4>
                    <p className="text-xs text-slate-500">Core modules included. No surprise unlock fees.</p>
                  </div>
                </div>
              </div>
            </div>
            <div className="bg-slate-50 dark:bg-black/20 p-12 md:p-16 flex flex-col justify-center items-center text-center border-l border-slate-100 dark:border-white/5">
              <h3 className="text-2xl font-bold text-brand-navy dark:text-white mb-4">Ready to scale?</h3>
              <p className="text-slate-500 text-sm mb-8">Get a custom walkthrough of the platform tailored to your business process.</p>
              <a href="/book-demo" className="w-full">
                <Button variant="primary" fullWidth className="mb-4">Book Demo</Button>
              </a>
              <a href="/contact" className="w-full">
                <Button variant="outline" fullWidth>Contact Sales</Button>
              </a>
            </div>
          </div>
        </div>
      </section>

      {/* 13. FINAL CTA */}
      <section className="py-32 relative overflow-hidden bg-brand-blue text-white">
        <div className="absolute inset-0 bg-[url('https://grainy-gradients.vercel.app/noise.svg')] opacity-20 mix-blend-multiply" />
        <div className="absolute top-[-50%] left-[-20%] w-[1000px] h-[1000px] bg-white/10 rounded-full blur-[150px]" />

        <div className="max-w-4xl mx-auto text-center relative z-10 px-4">
          <h2 className="text-5xl md:text-7xl font-bold mb-8 tracking-tight">
            Interested in Growing <br /> Your Business?
          </h2>
          <p className="text-xl text-blue-100 mb-12 max-w-2xl mx-auto leading-relaxed">
            Unify your CRM and ERP, boost productivity, and give management the clarity they deserve. It's time to systemize your SME.
          </p>
          <div className="flex flex-col sm:flex-row items-center justify-center gap-6">
            <a href="/book-demo">
              <motion.button
                whileHover={{ scale: 1.05 }}
                whileTap={{ scale: 0.95 }}
                className="px-10 py-5 bg-white text-brand-blue font-bold text-xl rounded-full shadow-2xl hover:shadow-[0_0_40px_rgba(255,255,255,0.4)] transition-all"
              >
                Book Demo
              </motion.button>
            </a>
            <a href="/contact" className="font-bold text-lg hover:text-blue-200 flex items-center gap-2 group">
              Contact Sales <ArrowRight size={20} className="group-hover:translate-x-1 transition-transform" />
            </a>
          </div>
        </div>
      </section>

    </div>
  );
};
